"use strict"

/**** Configuration variables: canvas & margins ****/
let svgWidth = 800;
let svgHeight = 600;

let leftMargin = 150;
let rightMargin = 40;
let topMargin = 40;
let bottomMargin = 80;

/**** Set up canvas and border ****/
let svg = d3.select("#canvas")
    .append("svg")
    .attr("width", svgWidth)
    .attr("height", svgHeight)

svg.append("rect")
    .attr("fill", "none")
    .attr("stroke", "black")
    .attr("width", svgWidth)
    .attr("height", svgHeight)

/* REMINDER: Our dataset is very long, so we're loading it from the bottom of the page. Soon you'll learn how to load it from an external file! */
let dataset = fillInData()

/**** Configuration variables: data ****/
let maxTemp = d3.max (dataset, function(value) {
    let tempsList = value.temps
    let biggestInList = tempsList.reduce(function (result, value) {
        return Math.max(result, value)
    },0)
    return biggestInList
})

let minTemp = d3.min (dataset, function(value) {
    return value.temps.reduce(function (result, value) {
        return Math.min(result, value)
    }, Infinity)
})

/**** Set up scales ****/

/* Figure out our X axis values */
let xAxisValues = Array.from(new Set(dataset.map(function (value) {
    return value.month
})))

let xScale = d3.scaleBand()
    .domain(xAxisValues)
    .range([leftMargin, svgWidth - rightMargin])
    .padding(.05)

let yAxisValues = Array.from(new Set(dataset.map(function (value) {
    return value.city
})))

// Sort city names reverse alphabetical
yAxisValues.sort(function (a, b) {
    if (a.toLowerCase() < b.toLowerCase()) {
        return 1
    }
    return -1
})

let yScale = d3.scaleBand()
    .domain(yAxisValues)
    .range([svgHeight - bottomMargin, topMargin])
    .padding(.05)

let colorScale = d3.scaleSequential()
    .interpolator(d3.interpolateInferno)
    .domain([maxTemp, minTemp])

let temperatures = svg.selectAll("rect.temp")
    .data(dataset).join("rect").classed("temp", true).attr("width", xScale.bandwidth()).attr("height", yScale.bandwidth()).attr("x", function (value) { return xScale(value.month) }).attr("y", function (value) { return yScale(value.city) }).attr("rx", 4).attr("ry", 4)
    .attr("fill", function (value) {

        //1.  get our temps list (Array) from current datum being processed by D3 (value)

        let tempsList = value.temps

        //2. use Array.reduce() to add together all of the values in the Array

        let result = tempsList.reduce(function (result, value) {
            return result + value
        },0)

        //3. Divide the total (the result of reduce) by the number of items in the Array

        let avgTemp = result / tempsList.length

        return colorScale(avgTemp)
    })

/**** label the axes ****/
let xAxisLabel = svg.append("text")
    .attr("x", svgWidth / 2)
    .attr("y", svgHeight - (bottomMargin / 3))
    .attr("text-anchor", "middle")
    .text("Months in 1999");

let yAxisLabel = svg.append("text")
    .attr("x", -svgHeight / 2)
    .attr("y", leftMargin / 4)
    .attr("text-anchor", "middle")
    .attr("alignment-baseline", "middle")
    .text("Temperature (Fahrenheit)")
    .attr("transform", "rotate(-90)");

/**** label key graph points ****/

let xAxis = svg.append("g")
    .attr("transform", `translate(0, ${svgHeight - bottomMargin})`)
    .call(d3.axisBottom().scale(xScale).tickSize(0))

/* Generate Y axis using D3 method */
let yAxis = svg.append("g")
    .attr("transform", `translate(${leftMargin}, 0)`)
    .call(d3.axisLeft().scale(yScale).tickSize(0))

/**** data below this line ****/

function fillInData() {
    /* CHANGE: Some months and cities now have a lot of temperature measurements! */

    return [{
        "city": "New York",
        "month": "Jan",
        "temps": [26, 37, 38, 33, 31, 31, 26, 37, 32, 30, 29, 38, 28, 33, 35, 39]
    },
    {
        "city": "New York",
        "month": "Feb",
        "temps": [29, 36, 32, 29, 35, 41, 33, 30, 35, 29, 31, 38, 37, 33, 34, 42]
    },
    {
        "city": "New York",
        "month": "Mar",
        "temps": [35, 47, 38, 39, 36, 49, 46, 49, 44, 43, 38, 37, 48, 39, 47, 50]
    },
    {
        "city": "New York",
        "month": "Apr",
        "temps": [44, 51, 46, 59, 53, 51, 51, 57, 50, 59, 49, 46, 46, 47, 48, 60]
    },
    {
        "city": "New York",
        "month": "May",
        "temps": [55, 55, 69, 67, 64, 68, 65, 57, 65, 55, 65, 60, 55, 69, 55, 71]
    },
    {
        "city": "New York",
        "month": "Jun",
        "temps": [64, 71, 78, 73, 75, 69, 68, 75, 72, 69, 66, 70, 76, 68, 67, 79]
    },
    {
        "city": "New York",
        "month": "Jul",
        "temps": [70, 85]
    },
    {
        "city": "New York",
        "month": "Aug",
        "temps": [69, 83]
    },
    {
        "city": "New York",
        "month": "Sep",
        "temps": [61, 76]
    },
    {
        "city": "New York",
        "month": "Oct",
        "temps": [50, 65]
    },
    {
        "city": "New York",
        "month": "Nov",
        "temps": [41, 54]
    },
    {
        "city": "New York",
        "month": "Dec",
        "temps": [32, 44]
    },
    {
        "city": "Los Angeles",
        "month": "Jan",
        "temps": [51, 56, 59, 54, 56, 66, 66, 61, 56, 59, 52, 55, 64, 59, 60, 63, 67]
    },
    {
        "city": "Los Angeles",
        "month": "Feb",
        "temps": [51, 52, 59, 58, 66, 56, 67]
    },
    {
        "city": "Los Angeles",
        "month": "Mar",
        "temps": [51, 67]
    },
    {
        "city": "Los Angeles",
        "month": "Apr",
        "temps": [53, 69]
    },
    {
        "city": "Los Angeles",
        "month": "May",
        "temps": [56, 70]
    },
    {
        "city": "Los Angeles",
        "month": "Jun",
        "temps": [58, 73]
    },
    {
        "city": "Los Angeles",
        "month": "Jul",
        "temps": [62, 69, 63, 66, 70, 63, 73, 67, 75, 75, 70, 69, 68, 65, 71, 68, 66, 68, 73, 70, 76, 69, 71, 71, 73, 70, 76, 69, 65, 66, 77]
    },
    {
        "city": "Los Angeles",
        "month": "Aug",
        "temps": [62, 79]
    },
    {
        "city": "Los Angeles",
        "month": "Sep",
        "temps": [62, 78]
    },
    {
        "city": "Los Angeles",
        "month": "Oct",
        "temps": [59, 75]
    },
    {
        "city": "Los Angeles",
        "month": "Nov",
        "temps": [55, 71]
    },
    {
        "city": "Los Angeles",
        "month": "Dec",
        "temps": [51, 67]
    },
    {
        "city": "Chicago",
        "month": "Jan",
        "temps": [18, 25, 22, 23, 19, 18, 19, 22, 32]
    }, {
        "city": "Chicago",
        "month": "Feb",
        "temps": [21, 36]
    }, {
        "city": "Chicago",
        "month": "Mar",
        "temps": [30, 46]
    }, {
        "city": "Chicago",
        "month": "Apr",
        "temps": [41, 59]
    },
    {
        "city": "Chicago",
        "month": "May",
        "temps": [52, 70]
    },
    {
        "city": "Chicago",
        "month": "Jun",
        "temps": [63, 81]
    },
    {
        "city": "Chicago",
        "month": "Jul",
        "temps": [68, 84]
    },
    {
        "city": "Chicago",
        "month": "Aug",
        "temps": [66, 82]
    },
    {
        "city": "Chicago",
        "month": "Sep",
        "temps": [57, 75]
    },
    {
        "city": "Chicago",
        "month": "Oct",
        "temps": [46, 63]
    },
    {
        "city": "Chicago",
        "month": "Nov",
        "temps": [34, 48]
    },
    {
        "city": "Chicago",
        "month": "Dec",
        "temps": [23, 36]
    }, {
        "city": "Houston",
        "month": "Jan",
        "temps": [44, 62]
    }, {
        "city": "Houston",
        "month": "Feb",
        "temps": [46, 62, 61, 48, 57, 61, 59, 64, 57, 59, 61, 54, 61, 53, 64, 50, 57, 56, 56, 56, 47, 48, 49, 56, 46, 56, 57, 65]
    }, {
        "city": "Houston",
        "month": "Mar",
        "temps": [54, 72]
    }, {
        "city": "Houston",
        "month": "Apr",
        "temps": [60, 78]
    }, {
        "city": "Houston",
        "month": "May",
        "temps": [66, 84]
    }, {
        "city": "Houston",
        "month": "Jun",
        "temps": [72, 90]
    }, {
        "city": "Houston",
        "month": "Jul",
        "temps": [74, 92]
    }, {
        "city": "Houston",
        "month": "Aug",
        "temps": [74, 93]
    }, {
        "city": "Houston",
        "month": "Sep",
        "temps": [70, 88]
    }, {
        "city": "Houston",
        "month": "Oct",
        "temps": [61, 81]
    }, {
        "city": "Houston",
        "month": "Nov",
        "temps": [52, 71]
    }, {
        "city": "Houston",
        "month": "Dec",
        "temps": [45, 63]
    }, {
        "city": "Phoenix",
        "month": "Jan",
        "temps": [46, 67]
    }, {
        "city": "Phoenix",
        "month": "Feb",
        "temps": [49, 71]
    }, {
        "city": "Phoenix",
        "month": "Mar",
        "temps": [53, 77]
    }, {
        "city": "Phoenix",
        "month": "Apr",
        "temps": [60, 85]
    }, {
        "city": "Phoenix",
        "month": "May",
        "temps": [69, 95]
    }, {
        "city": "Phoenix",
        "month": "Jun",
        "temps": [78, 104]
    }, {
        "city": "Phoenix",
        "month": "Jul",
        "temps": [83, 106]
    }, {
        "city": "Phoenix",
        "month": "Aug",
        "temps": [83, 104]
    }, {
        "city": "Phoenix",
        "month": "Sep",
        "temps": [77, 100]
    }, {
        "city": "Phoenix",
        "month": "Oct",
        "temps": [65, 89]
    }, {
        "city": "Phoenix",
        "month": "Nov",
        "temps": [53, 76]
    }, {
        "city": "Phoenix",
        "month": "Dec",
        "temps": [45, 66]
    }, {
        "city": "Philadelphia",
        "month": "Jan",
        "temps": [26, 40]
    }, {
        "city": "Philadelphia",
        "month": "Feb",
        "temps": [28, 44]
    }, {
        "city": "Philadelphia",
        "month": "Mar",
        "temps": [34, 53]
    }, {
        "city": "Philadelphia",
        "month": "Apr",
        "temps": [44, 64]
    }, {
        "city": "Philadelphia",
        "month": "May",
        "temps": [54, 74]
    }, {
        "city": "Philadelphia",
        "month": "Jun",
        "temps": [64, 83]
    }, {
        "city": "Philadelphia",
        "month": "Jul",
        "temps": [69, 87]
    }, {
        "city": "Philadelphia",
        "month": "Aug",
        "temps": [68, 85]
    }, {
        "city": "Philadelphia",
        "month": "Sep",
        "temps": [60, 78]
    }, {
        "city": "Philadelphia",
        "month": "Oct",
        "temps": [48, 67]
    }, {
        "city": "Philadelphia",
        "month": "Nov",
        "temps": [39, 56]
    }, {
        "city": "Philadelphia",
        "month": "Dec",
        "temps": [30, 45]
    }, {
        "city": "San Antonio",
        "month": "Jan",
        "temps": [39, 62]
    }, {
        "city": "San Antonio",
        "month": "Feb",
        "temps": [42, 67]
    }, {
        "city": "San Antonio",
        "month": "Mar",
        "temps": [50, 74]
    }, {
        "city": "San Antonio",
        "month": "Apr",
        "temps": [57, 80]
    }, {
        "city": "San Antonio",
        "month": "May",
        "temps": [65, 86]
    }, {
        "city": "San Antonio",
        "month": "Jun",
        "temps": [72, 91]
    }, {
        "city": "San Antonio",
        "month": "Jul",
        "temps": [74, 95]
    }, {
        "city": "San Antonio",
        "month": "Aug",
        "temps": [74, 95]
    }, {
        "city": "San Antonio",
        "month": "Sep",
        "temps": [69, 90]
    }, {
        "city": "San Antonio",
        "month": "Oct",
        "temps": [59, 82]
    }, {
        "city": "San Antonio",
        "month": "Nov",
        "temps": [49, 71]
    }, {
        "city": "San Antonio",
        "month": "Dec",
        "temps": [41, 64]
    }, {
        "city": "San Diego",
        "month": "Jan",
        "temps": [49, 65]
    }, {
        "city": "San Diego",
        "month": "Feb",
        "temps": [51, 65]
    }, {
        "city": "San Diego",
        "month": "Mar",
        "temps": [53, 66]
    }, {
        "city": "San Diego",
        "month": "Apr",
        "temps": [56, 67]
    }, {
        "city": "San Diego",
        "month": "May",
        "temps": [59, 69]
    }, {
        "city": "San Diego",
        "month": "Jun",
        "temps": [62, 71]
    }, {
        "city": "San Diego",
        "month": "Jul",
        "temps": [65, 75]
    }, {
        "city": "San Diego",
        "month": "Aug",
        "temps": [67, 76]
    }, {
        "city": "San Diego",
        "month": "Sep",
        "temps": [65, 76]
    }, {
        "city": "San Diego",
        "month": "Oct",
        "temps": [61, 73]
    }, {
        "city": "San Diego",
        "month": "Nov",
        "temps": [54, 69]
    }, {
        "city": "San Diego",
        "month": "Dec",
        "temps": [48, 65]
    }, {
        "city": "Dallas",
        "month": "Jan",
        "temps": [30, 57]
    }, {
        "city": "Dallas",
        "month": "Feb",
        "temps": [35, 61]
    }, {
        "city": "Dallas",
        "month": "Mar",
        "temps": [42, 69]
    }, {
        "city": "Dallas",
        "month": "Apr",
        "temps": [51, 77]
    }, {
        "city": "Dallas",
        "month": "May",
        "temps": [60, 84]
    }, {
        "city": "Dallas",
        "month": "Jun",
        "temps": [67, 91]
    }, {
        "city": "Dallas",
        "month": "Jul",
        "temps": [71, 95]
    }, {
        "city": "Dallas",
        "month": "Aug",
        "temps": [71, 96]
    }, {
        "city": "Dallas",
        "month": "Sep",
        "temps": [64, 89]
    }, {
        "city": "Dallas",
        "month": "Oct",
        "temps": [52, 80]
    }, {
        "city": "Dallas",
        "month": "Nov",
        "temps": [43, 68]
    }, {
        "city": "Dallas",
        "month": "Dec",
        "temps": [32, 58]
    }, {
        "city": "San Jose",
        "month": "Jan",
        "temps": [42, 60]
    }, {
        "city": "San Jose",
        "month": "Feb",
        "temps": [45, 64]
    }, {
        "city": "San Jose",
        "month": "Mar",
        "temps": [47, 67]
    }, {
        "city": "San Jose",
        "month": "Apr",
        "temps": [49, 70]
    }, {
        "city": "San Jose",
        "month": "May",
        "temps": [53, 75]
    }, {
        "city": "San Jose",
        "month": "Jun",
        "temps": [56, 81]
    }, {
        "city": "San Jose",
        "month": "Jul",
        "temps": [58, 84]
    }, {
        "city": "San Jose",
        "month": "Aug",
        "temps": [58, 84]
    }, {
        "city": "San Jose",
        "month": "Sep",
        "temps": [57, 82]
    }, {
        "city": "San Jose",
        "month": "Oct",
        "temps": [53, 76]
    }, {
        "city": "San Jose",
        "month": "Nov",
        "temps": [46, 67]
    }, {
        "city": "San Jose",
        "month": "Dec",
        "temps": [42, 61]
    }
    ]
}
