"use strict"

/**** Configuration variables: drawing area ****/
let svgWidth = 800
let svgHeight = 600

/* Configuration variables: margins */
let leftMargin = 80
let rightMargin = 25
let topMargin = 100
let bottomMargin = 60

/**** CANVAS SETUP ****/
/* Create canvas and center on page*/
d3.select("#container")
    .style("width", String(svgWidth) + "px")

let svg = d3.select("#drawing")
    .append("svg")
    .attr("width", svgWidth)
    .attr("height", svgHeight)

/* Draw canvas border */
svg.append("rect")
    .attr("fill", "none")
    .attr("stroke", "black")
    .attr("width", svgWidth)
    .attr("height", svgHeight)

/**** DEFINE DATASET (fold this code to save space) ****/
let dataset = [{
    area: "Back Bay",
    rent: 2920,
    income: 97.8,
    population: 218.8,
    charles: true
},
{
    area: "Charlestown",
    rent: 2585,
    income: 94.6,
    population: 164.4,
    charles: false
},
{
    area: "Chinatown",
    rent: 3210,
    income: 121.0,
    population: 68.7,
    charles: false
},
{
    area: "Downtown",
    rent: 3325,
    income: 168.6,
    population: 19.8,
    charles: true
},
{
    area: "Fenway",
    rent: 2400,
    income: 37.9,
    population: 217.7,
    charles: true
},
{
    area: "Mission Hill",
    rent: 2100,
    income: 37.3,
    population: 139.3,
    charles: false
},
{
    area: "North End",
    rent: 2350,
    income: 98.5,
    population: 106.1,
    charles: false
},
{
    area: "Allston",
    rent: 1900,
    income: 52.1,
    population: 288.2,
    charles: true
},
{
    area: "Brighton",
    rent: 1950,
    income: 61.7,
    population: 459.8,
    charles: true
},
{
    area: "Dorchester",
    rent: 1900,
    income: 62.2,
    population: 607.9,
    charles: false
},
{
    area: "Jamaica Plain",
    rent: 1950,
    income: 84.0,
    population: 412.6,
    charles: false
},
{
    area: "Roxbury",
    rent: 1700,
    income: 32.3,
    population: 525.3,
    charles: false
},
{
    area: "South Boston",
    rent: 2840,
    income: 89.1,
    population: 336.9,
    charles: false
},
{
    area: "West Roxbury",
    rent: 1810,
    income: 90.5,
    population: 304.4,
    charles: true
},
];

/**** Configuration variables: Data ****/
/* While not as convenient as having all these variables at the top of the program, if we write variables related to our data in a place where the data will already be loaded, we can then obtain these values from the data! */

let populationMin = d3.min(dataset, function (value) { return value.population })
let populationMax = d3.max(dataset, function (value) { return value.population })

/**** PREPROCESS DATA ****/
dataset.sort(function (a, b) {
    if (b.population > a.population) {
        return 1
    }
    return -1
})

/**** SET UP SCALE FUNCTIONS ****/

let yScale = d3.scaleLinear()
    .domain([0, 3500])
    .range([svgHeight - bottomMargin, topMargin])

let xScale = d3.scaleLinear()
    .domain([0, 180])
    .range([leftMargin, svgWidth - rightMargin])

let rScale = d3.scaleSqrt()
    .domain([0, 650])
    .range([1, 30])

/**** DRAW POINTS FOR SCATTER PLOT: ****/

let circles = svg.selectAll("circle")
    .data(dataset)
    .join("circle")
    .attr("cx", function (value) {
        return xScale(value.income)
    })
    .attr("cy", function (value) {
        return yScale(value.rent)
    })
    .attr("r", function (value) {
        return rScale(value.population)
    })
    .attr("fill", function (value) {
        if (value.charles) {
            return "lightblue"
        } else {
            return "pink"
        }
    })

/**** Draw X AND Y AXIS ELEMENTS ****/

let xAxis = svg.append("g")
    .classed("key", true)
    .attr("transform", `translate(0, ${svgHeight - bottomMargin})`)
    .call(d3.axisBottom()
        .scale(xScale))

let yAxis = svg.append("g")
    .classed("key", true)
    .attr("transform", `translate(${leftMargin}, 0)`)
    .call(d3.axisLeft()
        .scale(yScale))

/* DRAW AXIS LABELS */

let xAxisLabel = svg.append("text")
    .attr("class", "axisLabel")
    .attr("x", svgWidth / 2)
    .attr("y", svgHeight - bottomMargin / 4)
    .style("text-anchor", "middle")
    .text("Median Income (Thousands of Dollars)")

let yAxisLabel = svg.append("text")
    .attr("class", "axisLabel")
    .attr("transform", "rotate(-90)")
    .attr("x", -svgHeight / 2)
    .attr("y", leftMargin / 4)
    .style("text-anchor", "middle")
    .text("Median Rent")

/* MANUALLY ADD MAP LEGEND ELEMENTS */

let colorKey = svg.append("g")

colorKey.append("text")
    .text("Neighborhood Access to Charles")
    .attr("x", 120)
    .attr("y", 20)
    .classed("key", true)
    .classed("keyTitle", true)

colorKey.append("rect")
    .attr("width", 20)
    .attr("height", 20)
    .attr("fill", "lightblue")
    .attr("x", 20)
    .attr("y", 30)

colorKey.append("rect")
    .attr("width", 20)
    .attr("height", 20)
    .attr("fill", "pink")
    .attr("x", 20)
    .attr("y", 60)

colorKey.append("text")
    .text("Bordering Charles River")
    .attr("x", 50)
    .attr("y", 40)
    .classed("key", true)
    .classed("keyLabel", true)

colorKey.append("text")
    .text("Not on Charles River")
    .attr("x", 50)
    .attr("y", 70)
    .classed("key", true)
    .classed("keyLabel", true)

colorKey.attr("transform", "translate(560,10)")

colorKey.append("rect")
    .attr("width", 240)
    .attr("height", 100)
    .attr("fill", "none")
    .attr("stroke", "black")
    .attr("x", -5)
    .attr("y", -5)

let lowKey = svg.append("circle")
    .attr("fill", "grey")
    .attr("cx", 280)
    .attr("cy", 30 + rScale(populationMax))
    .attr("r", rScale(populationMin))

let lowKeyLabel = svg.append("text")
    .text(populationMin)
    .attr("x", Number(lowKey.attr("cx")) + Number(lowKey.attr("r")) + 5)
    .attr("y", 30 + rScale(populationMax))
    .classed("key", true)
    .classed("keyLabel", true)

let highKey = svg.append("circle")
    .attr("fill", "grey")
    .attr("cx", Number(lowKeyLabel.attr("x")) + lowKeyLabel.node().getComputedTextLength() + rScale(populationMax) + 20)
    .attr("cy", 30 + rScale(populationMax))
    .attr("r", rScale(populationMax))

let highKeyLabel = svg.append("text")
    .text(populationMax)
    .attr("x", Number(highKey.attr("cx")) + Number(highKey.attr("r")) + 5)
    .attr("y", 30 + rScale(populationMax))
    .classed("key", true)
    .classed("keyLabel", true)

svg.append("text")
    .text("Population in thousands")
    .attr("x", 350)
    .attr("y", 20)
    .classed("key", true)
    .classed("keyTitle", true)

/**** LABEL SCATTERPLOT POINTS */
/* TO DO: Make this code behave using a CSS class */

let pointLabels = svg.selectAll("text.label")
    .data(dataset)
    .join("text")
    .classed("label", true)
    .text(function (value) {
        return value.area
    })
    .attr("x", function (value) {
        return xScale(value.income)
    })
    .attr("y", function (value) {
        return yScale(value.rent) + rScale(value.population)
    })